/* ***************************************************************** 
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2004 Sandia Corporation and Argonne National
    Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
    with Sandia Corporation, the U.S. Government retains certain 
    rights in this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License 
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
    pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov      
   
  ***************************************************************** */
/*!
  \file   ObjectiveFunction.cpp
  \brief  

  \author Michael Brewer
  \author Thomas Leurent
  
  \date   2002-08-02
*/

#include "ObjectiveFunction.hpp"
#include "MsqVertex.hpp"
#include "MsqDebug.hpp"
#include "PatchData.hpp"
#include "MsqError.hpp"
#include "MsqHessian.hpp"
#include "SymMatrix3D.hpp"
#include <memory>  // auto_ptr

namespace MESQUITE_NS {



/*!Returns an appropiate value (eps) to use as a delta step for
  MsqVertex vertex in dimension k (i.e. k=0 -> x, k=1 -> y, k=2 -> z).
  The objective function value at the perturbed vertex position is given
  in local_val.
*/
double ObjectiveFunction::get_eps( PatchData &pd, 
                                   EvalType type,
                                   double &local_val,
                                   int dim,
                                   size_t vertex_index, 
                                   MsqError& err)
{
  double eps = 1.e-07;
  const double rho = 0.5;
  const int imax = 50;
  bool feasible = false;
  double tmp_var = 0.0;
  Vector3D delta(0,0,0);
  for (int i = 0; i < imax; ++i)
  {
    //perturb kth coord val and check feas if needed
    tmp_var = pd.vertex_by_index(vertex_index)[dim];
    delta[dim] = eps;
    pd.move_vertex( delta, vertex_index, err );
    feasible = evaluate( type, pd, local_val, OF_FREE_EVALS_ONLY, err ); MSQ_ERRZERO(err);
    //revert kth coord val
    delta = pd.vertex_by_index(vertex_index);
    delta[dim] = tmp_var;
    pd.set_vertex_coordinates( delta, vertex_index, err );
    //if step was too big, shorten it and go again
    if (feasible)
      return eps;
    else
      eps *= rho;
    //printf("new eps = %.30lf\n", eps);
  }//end while looking for feasible eps
  //printf("returning 0 eps\n");
  //printf("returning 0 eps\n");
  return 0.0;
}//end function get_eps

bool ObjectiveFunction::compute_subpatch_numerical_gradient(
                                 EvalType type,
                                 EvalType subtype,
                                 PatchData& pd,
                                 double& flocal,
                                 Vector3D& grad,
                                 MsqError& err )
{
  assert( pd.num_free_vertices() == 1 );
  
  double flocald=0;
  double eps=0;
  
  bool b = evaluate( type, pd, flocal, OF_FREE_EVALS_ONLY, err );
  if(MSQ_CHKERR(err) || !b) {
    return false;
  }

    //loop over the three coords x,y,z
  for(int j=0; j<3; ++j) {
    eps = get_eps( pd, subtype, flocald, j, 0, err ); MSQ_ERRZERO(err);
    if (eps==0) {
      /*MSQ_SETERR(err)("Dividing by zero in Objective Functions numerical grad",
                      MsqError::INVALID_STATE);
      return false;*/
	grad[j]=0;
    }
    else
        grad[j]=(flocald-flocal)/eps;
  }

/*****************************NEW************************************************/
/*MsqVertex vertex = pd.vertex_by_index(0);
if(vertex[2] == 0 || vertex[2] == 1)
  {
  	grad[2]=0;
	//printf("here we are\n");
  }
  
	Vector3D points[20][3];
	points[0][0].set(1,0,1);
	points[0][1].set(1,0,0);
	points[0][2].set(0.404509,0.293893,1.000000);
	
	points[10][0].set(0.404509,0.293893,0.000000);
	points[10][1].set(1,0,0);
	points[10][2].set(0.404509,0.293893,1.000000);
	
	points[1][0].set(0.309017, 0.951057, 1.000000);
	points[1][1].set(0.309017, 0.951057, 0.000000);
	points[1][2].set(0.404509, 0.293893, 1.000000);
	
	points[11][0].set(0.404509, 0.293893, 0.000000);
	points[11][1].set(0.309017, 0.951057, 0.000000);
	points[11][2].set(0.404509, 0.293893, 1.000000);
	
	points[2][0].set(-0.809017, 0.587785, 1.000000);
	points[2][1].set(-0.809017, 0.587785, 0.000000);
	points[2][2].set(-0.154509, 0.475528, 1.000000);
	
	points[12][0].set(-0.154509, 0.475528, 0.000000);
	points[12][1].set(-0.809017, 0.587785, 0.000000);
	points[12][2].set(-0.154509, 0.475528, 1.000000);
	
	points[3][0].set(-0.809017, 0.587785, 1.000000);
	points[3][1].set(-0.809017, 0.587785, 0.000000);
	points[3][2].set(-0.500000, 0.000000, 1.000000);
	
	points[13][0].set(-0.500000, 0.000000, 0.000000);
	points[13][1].set(-0.809017, 0.587785, 0.000000);
	points[13][2].set(-0.500000, 0.000000, 1.000000);
	
	points[4][0].set(0.309017, 0.951057, 1.000000);
	points[4][1].set(0.309017, 0.951057, 0.000000);
	points[4][2].set(-0.154509, 0.475528, 1.000000);
	
	points[14][0].set(-0.154509, 0.475528, 0.000000);
	points[14][1].set(0.309017, 0.951057, 0.000000);
	points[14][2].set(-0.154509, 0.475528, 1.000000);
	
	points[5][0].set(-0.809017, -0.587785, 1.000000);
	points[5][1].set(-0.809017, -0.587785, 0.000000);
	points[5][2].set(-0.500000, 0.000000, 1.000000);
	
	points[15][0].set(-0.500000, 0.000000, 0.000000);
	points[15][1].set(-0.809017, -0.587785, 0.000000);
	points[15][2].set(-0.500000, 0.000000, 1.000000);
	
	points[6][0].set(-0.809017, -0.587785, 1.000000);
	points[6][1].set(-0.809017, -0.587785, 0.000000);
	points[6][2].set(-0.154509, -0.475528, 1.000000);
	
	points[16][0].set(-0.154509, -0.475528, 0.000000);
	points[16][1].set(-0.809017, -0.587785, 0.000000);
	points[16][2].set(-0.154509, -0.475528, 1.000000);
	
	points[7][0].set(-0.154509, -0.475528, 1.000000);
	points[7][1].set(-0.154509, -0.475528, 0.000000);
	points[7][2].set(0.309017, -0.951057, 1.000000);
	
	points[17][0].set(0.309017, -0.951057, 0.000000);
	points[17][1].set(-0.154509, -0.475528, 0.000000);
	points[17][2].set(0.309017, -0.951057, 1.000000);
	
	points[8][0].set(0.309017, -0.951057, 1.000000);
	points[8][1].set(0.309017, -0.951057, 0.000000);
	points[8][2].set(0.404509, -0.293893, 1.000000);
	
	points[18][0].set(0.404509, -0.293893, 0.000000);
	points[18][1].set(0.309017, -0.951057, 0.000000);
	points[18][2].set(0.404509, -0.293893, 1.000000);
	
	points[9][0].set(0.404509, -0.293893, 1.000000);
	points[9][1].set(0.404509,-0.293893,0.000000);
	points[9][2].set(1,0,1);
	
	points[19][0].set(1,0,0);
	points[19][1].set(0.404509,-0.293893,0.000000);
	points[19][2].set(1,0,1);
  
  int count=0;
  for(int i=0; i<20; i++)
  {
	double vol=(points[i][1]-points[i][0])%((points[i][2]-points[i][0])*(vertex-points[i][0]));
	
	if (fabs(vol) < 1e-2)
	{
		if(i<10)
			{
				if(points[i][0][0]<points[i][2][0])
				{
					if(vertex[0]<points[i][0][0] ||vertex[0]>points[i][2][0])
						continue;
				}
				else
				{
					if(vertex[0]>points[i][0][0] ||vertex[0]<points[i][2][0])
						continue;
				}
			}
			else
			{
				if(points[i][0][0]<points[i][1][0])
				{
					if(vertex[0]<points[i][0][0] ||vertex[0]>points[i][1][0])
						continue;
				}
				else
				{
					if(vertex[0]>points[i][0][0] ||vertex[0]<points[i][1][0])
						continue;
				}
			}
		//Vector3D normal;
		//normal = (points[i][1]-points[i][0])*(points[i][2]-points[i][0]);
		//double orig  = grad%normal;
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		grad = (grad%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
		//printf("%lf %lf %lf\n", points[i][0][0], points[i][0][1], points[i][0][2]);
		//printf("%lf %lf %lf\n", points[i][1][0], points[i][1][1], points[i][1][2]);
		//printf("%lf %lf %lf\n", points[i][2][0], points[i][2][1], points[i][2][2]);
		//printf("%.10lf %.10lf %10lf\n", orig, grad%normal, orig/(grad%normal));	
		count++;
	} 
  }
  //if(count>2)
	//grad[0]=grad[1]=0;
  //if (count>0)
  //{
	//printf("%lf %lf %lf\n", grad[0], grad[1], grad[2]);
  //}
  //printf("%lf %lf %lf\n", grad[0], grad[1], grad[2]);*/
/*****************************NEW************************************************/

  return true;
}

bool ObjectiveFunction::compute_patch_numerical_gradient(  EvalType type,
                                                           EvalType subtype,
                                                           PatchData& pd,
                                                           double& flocal,
                                                           msq_std::vector<Vector3D>& grad,
                                                           MsqError& err )
{
  double flocald=0;
  double eps=0;
  
  bool b = evaluate( type, pd, flocal, OF_FREE_EVALS_ONLY, err );
  if(MSQ_CHKERR(err) || !b) {
    return false;
  }

  for (size_t i = 0; i < pd.num_free_vertices(); ++i) {
      //loop over the three coords x,y,z
    for(int j=0; j<3; ++j) {
      eps = get_eps( pd, subtype, flocald, j, i, err ); MSQ_ERRZERO(err);
      if (eps==0) {
        /*MSQ_SETERR(err)("Dividing by zero in Objective Functions numerical grad",
                        MsqError::INVALID_STATE);
        return false;*/
	grad[i][j] = 0;
      }
      grad[i][j]=(flocald-flocal)/eps;
    }
/*****************************NEW************************************************/
/*MsqVertex vertex = pd.vertex_by_index(i);
if(vertex[2] == 0 || vertex[2] == 1)
		grad[i][2]=0;
	
	Vector3D points[20][3];
	points[0][0].set(1,0,1);
	points[0][1].set(1,0,0);
	points[0][2].set(0.404509,0.293893,1.000000);
	
	points[10][0].set(0.404509,0.293893,0.000000);
	points[10][1].set(1,0,0);
	points[10][2].set(0.404509,0.293893,1.000000);
	
	points[1][0].set(0.309017, 0.951057, 1.000000);
	points[1][1].set(0.309017, 0.951057, 0.000000);
	points[1][2].set(0.404509, 0.293893, 1.000000);
	
	points[11][0].set(0.404509, 0.293893, 0.000000);
	points[11][1].set(0.309017, 0.951057, 0.000000);
	points[11][2].set(0.404509, 0.293893, 1.000000);
	
	points[2][0].set(-0.809017, 0.587785, 1.000000);
	points[2][1].set(-0.809017, 0.587785, 0.000000);
	points[2][2].set(-0.154509, 0.475528, 1.000000);
	
	points[12][0].set(-0.154509, 0.475528, 0.000000);
	points[12][1].set(-0.809017, 0.587785, 0.000000);
	points[12][2].set(-0.154509, 0.475528, 1.000000);
	
	points[3][0].set(-0.809017, 0.587785, 1.000000);
	points[3][1].set(-0.809017, 0.587785, 0.000000);
	points[3][2].set(-0.500000, 0.000000, 1.000000);
	
	points[13][0].set(-0.500000, 0.000000, 0.000000);
	points[13][1].set(-0.809017, 0.587785, 0.000000);
	points[13][2].set(-0.500000, 0.000000, 1.000000);
	
	points[4][0].set(0.309017, 0.951057, 1.000000);
	points[4][1].set(0.309017, 0.951057, 0.000000);
	points[4][2].set(-0.154509, 0.475528, 1.000000);
	
	points[14][0].set(-0.154509, 0.475528, 0.000000);
	points[14][1].set(0.309017, 0.951057, 0.000000);
	points[14][2].set(-0.154509, 0.475528, 1.000000);
	
	points[5][0].set(-0.809017, -0.587785, 1.000000);
	points[5][1].set(-0.809017, -0.587785, 0.000000);
	points[5][2].set(-0.500000, 0.000000, 1.000000);
	
	points[15][0].set(-0.500000, 0.000000, 0.000000);
	points[15][1].set(-0.809017, -0.587785, 0.000000);
	points[15][2].set(-0.500000, 0.000000, 1.000000);
	
	points[6][0].set(-0.809017, -0.587785, 1.000000);
	points[6][1].set(-0.809017, -0.587785, 0.000000);
	points[6][2].set(-0.154509, -0.475528, 1.000000);
	
	points[16][0].set(-0.154509, -0.475528, 0.000000);
	points[16][1].set(-0.809017, -0.587785, 0.000000);
	points[16][2].set(-0.154509, -0.475528, 1.000000);
	
	points[7][0].set(-0.154509, -0.475528, 1.000000);
	points[7][1].set(-0.154509, -0.475528, 0.000000);
	points[7][2].set(0.309017, -0.951057, 1.000000);
	
	points[17][0].set(0.309017, -0.951057, 0.000000);
	points[17][1].set(-0.154509, -0.475528, 0.000000);
	points[17][2].set(0.309017, -0.951057, 1.000000);
	
	points[8][0].set(0.309017, -0.951057, 1.000000);
	points[8][1].set(0.309017, -0.951057, 0.000000);
	points[8][2].set(0.404509, -0.293893, 1.000000);
	
	points[18][0].set(0.404509, -0.293893, 0.000000);
	points[18][1].set(0.309017, -0.951057, 0.000000);
	points[18][2].set(0.404509, -0.293893, 1.000000);
	
	points[9][0].set(0.404509, -0.293893, 1.000000);
	points[9][1].set(0.404509,-0.293893,0.000000);
	points[9][2].set(1,0,1);
	
	points[19][0].set(1,0,0);
	points[19][1].set(0.404509,-0.293893,0.000000);
	points[19][2].set(1,0,1);
	
	int count=0;
	for(int i=0; i<20; i++)
	{
		if(i<10)
		{
			if(points[i][0][0]<points[i][2][0])
			{
				if(vertex[0]+1e-6<points[i][0][0] ||vertex[0]-1e-6>points[i][2][0])
					continue;
			}
			else
			{
				if(vertex[0]-1e-6>points[i][0][0] ||vertex[0]+1e-6<points[i][2][0])
					continue;
			}
		}
		else
		{
			if(points[i][0][0]<points[i][1][0])
			{
				if(vertex[0]+1e-6<points[i][0][0] ||vertex[0]-1e-6>points[i][1][0])
					continue;
			}
			else
			{
				if(vertex[0]-1e-6>points[i][0][0] ||vertex[0]+1e-6<points[i][1][0])
					continue;
			}
		}
		double vol=(points[i][1]-points[i][0])%((points[i][2]-points[i][0])*(vertex-points[i][0]));
		if (fabs(vol) < 1e-2)
		{
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			grad[i] = (grad[i]%(points[i][1]-points[i][0]))*(points[i][1]-points[i][0])
			+ (grad[i]%(points[i][2]-points[i][0]))*(points[i][2]-points[i][0]);
			count++;
		}
	}
	if(count>2)
		grad[i][0]=grad[i][1]=0;*/
/*****************************NEW************************************************/
  }
  
  return true;
}
  


/*! 
  Numerically Calculates the gradient of the ObjectiveFunction for the
  free vertices in the patch.  Returns 'false' if the patch is outside
  of a required feasible region, returns 'ture' otherwise.
  The behavior of the function depends on the value of the boolean
  useLocalGradient.  If useLocalGradient is set to
  'true', compute_numerical_gradient creates a sub-patch around a free
  vertex, and then perturbs that vertex in one of the coordinate directions.
  Only the ObjectiveFunction value on the local sub-patch is used in the
  computation of the gradient.  Therefore, useLocalGradient should only
  be set to 'true' for ObjectiveFunctions which can use this method.  Unless
  the concrete ObjectiveFunction sets useLocalGradient to 'true' in its
  constructor, the value will be 'false'.  In this case, the objective
  function value for the entire patch is used in the calculation of the
  gradient.  This is computationally expensive, but it is numerically
  correct for all (C_1) functions.
  \param pd  PatchData on which the gradient is taken.
  \param grad  Array of Vector3D of length the number of vertices used to store gradient.
  \param OF_val will be set to the objective function value.
 */
bool ObjectiveFunction::evaluate_with_gradient( EvalType eval_type,
                                                PatchData &pd,
                                                double& OF_val,
                                                msq_std::vector<Vector3D>& grad,
                                                MsqError &err )
{
  bool b;
  grad.resize( pd.num_free_vertices() );
  
    // Fast path for single-free-vertex patch
  if (pd.num_free_vertices() == 1) {
    const EvalType sub_type = (eval_type == CALCULATE) ? CALCULATE : TEMPORARY;
    b = compute_subpatch_numerical_gradient( eval_type, sub_type, pd, OF_val, grad[0], err );
    return !MSQ_CHKERR(err) && b;
  }
  
  ObjectiveFunction* of = this;
  msq_std::auto_ptr<ObjectiveFunction> deleter;
  if (eval_type == CALCULATE) {
    of->clear();
    b = of->evaluate( ACCUMULATE, pd, OF_val, OF_FREE_EVALS_ONLY, err );
    if (err) { // OF doesn't support BCD type evals, try slow method
      err.clear();
      of->clear();
      b = compute_patch_numerical_gradient( CALCULATE, CALCULATE, pd, OF_val, grad, err );
      return !MSQ_CHKERR(err) && b;
    }
    else if (!b)
    {
      return b;
    }
  } 
  else {
    b = this->evaluate( eval_type, pd, OF_val, OF_FREE_EVALS_ONLY, err );
    if (MSQ_CHKERR(err) || !b)
      return false;
    of = this->clone();
    deleter = msq_std::auto_ptr<ObjectiveFunction>(of);
  }

    // Determine number of layers of adjacent elements based on metric type.
  unsigned layers = min_patch_layers();
  
    // Create a subpatch for each free vertex and use it to evaluate the
    // gradient for that vertex.
  double flocal;
  PatchData subpatch;
  for (size_t i = 0; i < pd.num_free_vertices(); ++i)
  {
    pd.get_subpatch( i, layers, subpatch, err ); MSQ_ERRZERO(err);
    b = of->compute_subpatch_numerical_gradient( SAVE, TEMPORARY, subpatch, flocal, grad[i], err );
    if (MSQ_CHKERR(err) || !b) {
      of->clear();
      return false;
    }
  }
  
  of->clear();
  return true;
}

bool ObjectiveFunction::evaluate_with_Hessian_diagonal( EvalType type, 
                                        PatchData& pd,
                                        double& value_out,
                                        msq_std::vector<Vector3D>& grad_out,
                                        msq_std::vector<SymMatrix3D>& hess_diag_out,
                                        MsqError& err )
{
  MsqHessian hess;
  hess.initialize( pd, err ); MSQ_ERRZERO(err);
  bool val = evaluate_with_Hessian( type, pd, value_out, grad_out, hess, err );
  MSQ_ERRZERO(err);
  hess_diag_out.resize( hess.size() );
  for (size_t i = 0; i < hess.size(); ++i)
    hess_diag_out[i] = hess.get_block(i,i)->upper();
  return val;
}

bool ObjectiveFunction::evaluate_with_Hessian( EvalType type, 
                                               PatchData& pd,
                                               double& value_out,
                                               msq_std::vector<Vector3D>& grad_out,
                                               MsqHessian& Hessian_out,
                                               MsqError& err ) 
{
      MSQ_SETERR(err)("No Hessian available for this objective function.\n"
                      "Choose either a different objective function or a "
                      "different solver.\n",
                    MsqError::INVALID_STATE);
      return false;
}



} // namespace Mesquite

